<?php
/**
 * Import sales orders into OrderWise
 *
 * System/Version:      Shopify
 * Developer:           Toby Loudon
 * OnTime Ref:          49248
 * Completed:           28/07/2020
 * Production URL:      http://cestevenson-orderwise.net/orderwise_shopify/orders.php
 * Staging URL:
 */

/** @var Settings $settings */
require './config.php';

// We also process the successfully imported order ids
//if ($_GET['action'] == 'imported') {
//    WriteToLog('posted order ids flag found in url');
//    $ecom->MarkOrdersImported($_POST);
//    exit;
//}

// Instantiate Shopify validate the store id from the url
try {
    $shopify = new Shopify($ecom->GetStoreId(), $settings);
    $shopify->getTaxTitles(); // 17/10/2018 | AC | Dev ID 39000 | Call method to fetch all tax codes (tax titles)

    // 19/05/2020 | CB | 49588 | Added lookups to use different description to that given by Shopify
    $delivery_method_lookup = new CsvLookupLoader('delivery_methods.csv');
    $payment_method_lookup = new CsvLookupLoader('payment_methods.csv');
    $stock_location_lookup = new CsvLookupLoader('stock_locations.csv');

} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}

// Pull the orders through for the first page
try {
    $order_filter = array();
    $order_filter['financial_status'] = 'paid';
    $order_filter['fulfillment_status'] = 'unshipped';
    $order_filter['status'] = 'open';
    $order_filter['updated_at_min'] = $ecom->GetOrdersFromDate();
    $order_filter['order_id'] = $_GET['order']; // if this is set other filters are ignored
    $order_list = $shopify->LoadOrders($order_filter);
} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}

// Create the XML headers which are sent if we drop out for no orders
$ecom->CreateOrderXmlHeaders();
if (empty($order_list)) {
    $ecom->ExportToXml();
    exit;
}

// Instantiate the string formatter
$format = new DataFormat();

// There is no mechanism in Shopify to mark orders as imported
// so we store them in a local database and cross check these with
// the order we're trying to import.
$imported_order_ids = $ecom->GetPreviouslyImportedOrderIds();

// A single page of orders is returned so process these.
// At the end of the process, we check for a next page
do {
    if (!empty($_GET['debug'])) {
        dd($order_list);
    }
    WriteToLog(count($order_list) . ' orders found for processing');
    foreach ($order_list as $foo => $order_row) {
        // Check we've not previously imported the order
        if (!is_numeric($_GET['order']) && isset($imported_order_ids[trim($order_row->id)])) {
            continue;
        }

        // Create new order details object
        $order_details = $ecom->NewOrder();

        // For the payment gateway we need to make a transaction call
        try {
            $transactions = $shopify->GetTransaction($order_row->id);
            if (is_array($transactions)) {
                foreach ($transactions as $k => $v) {
                    if ($v->status == 'success' && in_array(strtolower($v->kind), ['sale', 'capture'])) {
                        // 19/05/2020 | CB | Dev 49588 | Get payment description from lookup if set
                        $payment_description = $payment_method_lookup->getLookupValueIfExists($v->gateway);

                        $order_details->payments[] = array('description' => $payment_description, 'amount' => $v->amount);
                    }
                }
            }
        } catch (Exception $e) {
            continue;
        }

        // Basics
        $order_details->order_id = $order_row->id;
        $order_details->order_date = $order_row->created_at;
        $order_details->customer_order_ref = $order_row->order_number;

        // 04/07/2019 | DS | Bug 59530 | Clear the old address data so we don't populate address data for an order that doesn't have an address
        $s_addr = new stdClass();

        // Customer
        if (isset($order_row->billing_address)) {
            $s_addr = $order_row->billing_address;
        } else {
            if (isset($order_row->customer->default_address)) {
                $s_addr = $order_row->customer->default_address;
            } else {
                if (isset($order_row->shipping_address)) {
                    $s_addr = $order_row->shipping_address;
                }
            }
        }
        $order_details->customer_currency_code = $order_row->currency;
        $order_details->customer_account = $order_row->customer->id;
        $order_details->customer['statement_addr']->salutation = "";
        $order_details->customer['statement_addr']->name = $format->FormatName($s_addr->first_name . ' ' . $s_addr->last_name);
        $order_details->customer['statement_addr']->line1 = $format->FormatAddress($s_addr->address1);
        $order_details->customer['statement_addr']->line2 = $format->FormatAddress($s_addr->address2);
        $order_details->customer['statement_addr']->town = str_replace('— Please Select —', '', $format->FormatAddress($s_addr->city));
        $order_details->customer['statement_addr']->county = str_replace('— Please Select —', '', $format->FormatAddress($s_addr->province));
        $order_details->customer['statement_addr']->postcode = $format->FormatPostcode($s_addr->zip);
        $order_details->customer['statement_addr']->country = $s_addr->country;
        $order_details->customer['statement_addr']->countryiso = $s_addr->country_code;
        $order_details->customer['statement_addr']->telephone = $format->FormatTelephone($s_addr->phone);
        $order_details->customer['statement_addr']->fax = '';
        $order_details->customer['statement_addr']->email = $order_row->email;
        $order_details->customer['invoice_addr'] = clone $order_details->customer['statement_addr'];

        // Delivery
        $order_details->customer['delivery_addr']->name = $format->FormatName($order_row->shipping_address->company);
        $order_details->customer['delivery_addr']->contact = $format->FormatName($order_row->shipping_address->first_name . ' ' . $order_row->shipping_address->last_name);
        $order_details->customer['delivery_addr']->line1 = $format->FormatAddress($order_row->shipping_address->address1);
        $order_details->customer['delivery_addr']->line2 = $format->FormatAddress($order_row->shipping_address->address2);
        $order_details->customer['delivery_addr']->town = str_replace('— Please Select —', '', $format->FormatAddress($order_row->shipping_address->city));
        $order_details->customer['delivery_addr']->county = str_replace('— Please Select —', '', $format->FormatAddress($order_row->shipping_address->province));
        $order_details->customer['delivery_addr']->postcode = $format->FormatPostcode($order_row->shipping_address->zip);
        $order_details->customer['delivery_addr']->country = $order_row->shipping_address->country;
        $order_details->customer['delivery_addr']->countryiso = $order_row->shipping_address->country_code;
        $order_details->customer['delivery_addr']->telephone = $format->FormatTelephone($order_row->shipping_address->phone);
        $order_details->customer['delivery_addr']->email = $order_row->email;

        // Contact
        $order_details->customer['contact']->email = $order_row->customer->email;
        $order_details->customer['contact']->telephone = $order_row->customer->phone;

        // Delivery method
        $order_details->delivery_method = $delivery_method_lookup->getLookupValueIfExists($order_row->shipping_lines[0]->code);
        $order_details->order_special_instructions = $order_row->note;
        $order_details->delivery_special_instructions = '';

        // Shipping
        $shipping_tax = 0;
        $shipping_tax_rate = 0;
        if (count($order_row->shipping_lines) > 0) {
            foreach ($order_row->shipping_lines as $v) {
                $order_details->delivery_gross += $v->price;
                if (count($v->tax_lines) > 0) {
                    foreach ($v->tax_lines as $t) {
                        if ($shopify->isTaxTitle($t->title)) {
                            $shipping_tax += $t->price;
                            $shipping_tax_rate += $t->rate;
                        }
                    }
                }
            }
        }
        $order_details->delivery_vat = $shipping_tax;

        if (!$order_row->taxes_included) {
            $order_details->delivery_net = $order_details->delivery_gross;
            $order_details->delivery_gross += $shipping_tax;
        } else {
            $order_details->delivery_net = $order_details->delivery_gross - $shipping_tax;
        }
        if ($order_details->delivery_gross > 0) {
            $order_details->delivery_vat_code = $format->GetVatCode($order_details->delivery_vat);
        }

        // Financials
        $order_details->order_gross = $order_row->total_price;
        $order_details->order_vat = $order_row->total_tax;
        $order_details->order_net = $order_row->total_price - $order_row->total_tax;

        // Lines
        $dissur = ['T0' => 0, 'T1' => 0];
        foreach ($order_row->line_items as $line) {
            $line_tax_rate = 0;
            $line_tax = 0;
            foreach ($line->tax_lines as $k => $v) {
                if ($shopify->isTaxTitle($v->title)) {
                    $line_tax_rate += $v->rate;
                    $line_tax += $v->price;
                }
            }

            // Apply OnBuy-specific logic
            if (strpos($order_row->note, 'OnBuy Marketplace-Integration') !== false) {
                $tax_code = 'T1'; // Force "T1" tax code for OnBuy orders
                $vat_rate = 0.20;

                // Calculate the net price by excluding VAT (assume gross price includes VAT)
                $gross = $line->price;
                $net = $gross / (1 + $vat_rate); // Exclude 20% VAT
            } else {
                // For non-OnBuy orders, check if VAT is included and exclude it if needed
                $tax_code = $format->GetVatCode($line_tax);

                // Assume the price includes VAT and calculate net price
                if ($order_row->taxes_included) {
                    $vat_rate = 0.20; // Assume 20% VAT
                    $gross = $line->price;
                    $net = $gross / (1 + $vat_rate); // Exclude VAT to get net price
                } else {
                    // If VAT is not included, just use the price as net
                    $net = $line->price;
                }
            }

            // Set the item_net to reflect the unit price, not multiplied by quantity
            $order_details->order_lines[] = array(
                'ecommerce_code' => $line->sku,
                'variant_code' => $line->sku,
                'external_item_id' => $line->id,
                'quantity' => $line->quantity,
                'item_net' => $net, // Net price after excluding VAT
                'vat_code' => $tax_code,
                'stock_location_id' => isset($shopify->stock_locations[$order_row->location_id]) ? $shopify->stock_locations[$order_row->location_id]->name : ''
            );
        }

        // Add the order to the existing xml dom
        $max_orders_reached = false;
        if ($ecom->AppendOrderToXml($order_details)) {
            if ($ecom->GetOrderCounter() >= $ecom->GetMaximumOrdersPerSession() && $ecom->GetMaximumOrdersPerSession() != 0) {
                $max_orders_reached = true;
                break;
            }
        }
    }

    // If we've reached the max required orders flag more_orders
    // as false to drop out of the do...while
    if ($max_orders_reached || $_GET['order'] != '') {
        $more_orders = false;
    } else {
        try {
            $order_list = $shopify->LoadOrders($order_filter);
        } catch (Exception $e) {
            $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
            $ecom->ReturnResponseXML();
        }
        $more_orders = (count($order_list) > 0) ? true : false;
    }
} while ($more_orders);

$ecom->ExportToXml();
