<?php

/**
 * Class CsvLookupLoader
 *
 * Takes the contents of a csv file and turns into a lookup array
 * Expects two columns on the file, with a header
 * First column on the file be used as the key for the array, the second
 * will be used as the data
 *
 * Example file contents:
 * cart_payment_method, description_to_use
 * card_payment, Debit Card Payment
 * sagepay_form, SagePay
 */
class CsvLookupLoader
{
    private $lookup_filename;
    private $lookup_data;


    public function __construct($lookup_filename)
    {
        $this->lookup_filename = $lookup_filename;

        if (!file_exists($this->lookup_filename)) {
            throw new Exception("'{$this->lookup_filename}' file could not be found");
        }

        $this->lookup_data = $this->getLookupDataFromCSV();
    }


    private function getLookupDataFromCSV()
    {
        $csv_contents = array_map('str_getcsv', file($this->lookup_filename));
        array_shift($csv_contents); // Remove header column

        $lookup_data = [];
        foreach ($csv_contents as $csv_content) {
            $header = !empty($csv_content[0]) ? strtolower($csv_content[0]) : null;
            $data = !empty($csv_content[1]) ? $csv_content[1] : null;

            /**
             * Check for null in case the header/data happens to be 'falsey' value
             * so would be missed by using something like empty
             */
            if (is_null($header) || is_null($data)) {
                continue;
            }

            $lookup_data[$header] = trim($data);
        }

        return $lookup_data;
    }


    public function getLookupValueIfExists($value)
    {
        $key = strtolower($value);

        return !empty($this->lookup_data[$key]) ? $this->lookup_data[$key] : $value;
    }


    public function getKeyFromLookupValueIfExists($value)
    {
        // NOTE - did look at using array_search instead of below, but you will hit issues with casing

        if (empty($this->lookup_data)) {
            return $value;
        }

        foreach ($this->lookup_data as $k => $v) {
            if (strtolower($value) == strtolower($v)) {
                return $k;
            }
        }

        return $value;
    }
}