<?php
/**
 * Import sales orders into OrderWise
 *
 * System/Version:      Shopify
 * Developer:           Toby Loudon
 * OnTime Ref:          49248
 * Completed:           28/07/2020
 * Production URL:      http://cestevenson-orderwise.net/orderwise_shopify/orders.php
 * Staging URL:
 */

/** @var Settings $settings */
require './config.php';

// We also process the successfully imported order ids
//if ($_GET['action'] == 'imported') {
//    WriteToLog('posted order ids flag found in url');
//    $ecom->MarkOrdersImported($_POST);
//    exit;
//}


// Instantiate Shopify validate the store id from the url
try {
    $shopify = new Shopify($ecom->GetStoreId(), $settings);
    $shopify->getTaxTitles(); // 17/10/2018 | AC | Dev ID 39000 | Call method to fetch all tax codes (tax titles)

    // 19/05/2020 | CB | 49588 | Added lookups to use different description to that given by Shopify
    $delivery_method_lookup = new CsvLookupLoader('delivery_methods.csv');
    $payment_method_lookup = new CsvLookupLoader('payment_methods.csv');
    $stock_location_lookup = new CsvLookupLoader('stock_locations.csv');

} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}


// Pull the orders through for the first page
try {
    $order_filter = array();
    $order_filter['financial_status'] = 'paid';
    $order_filter['fulfillment_status'] = 'unshipped';
    $order_filter['status'] = 'open';
    $order_filter['updated_at_min'] = $ecom->GetOrdersFromDate();
    $order_filter['order_id'] = $_GET['order']; // if this is set other filters are ignored
    $order_list = $shopify->LoadOrders($order_filter);
} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}


// Create the XML headers which are sent if we drop out for no orders
$ecom->CreateOrderXmlHeaders();
if (empty($order_list)) {
    $ecom->ExportToXml();
    exit;
}


// Instantiate the string formatter
$format = new DataFormat();


// There is no mechanism in Shopify to mark orders as imported
// so we store them in a local database and cross check these with
// the order we're trying to import.
$imported_order_ids = $ecom->GetPreviouslyImportedOrderIds();


// A single page of orders is returned so process these.
// At the end of the process we check for a next page
do {
    if (!empty($_GET['debug'])) {
        dd($order_list);
    }
    WriteToLog(count($order_list) . ' orders found for processing');
    foreach ($order_list as $foo => $order_row) {
        // Check we've not previously imported the order
        if (!is_numeric($_GET['order']) && isset($imported_order_ids[trim($order_row->id)])) {
            continue;
        }

        // Create new order details object
        $order_details = $ecom->NewOrder();

        // For the payment gateway we need to make a transaction call
        try {
            $transactions = $shopify->GetTransaction($order_row->id);
            if (is_array($transactions)) {
                foreach ($transactions as $k => $v) {
                    // Need to check for $v->receipt->paid_amount as this indicates the completed payment as opposed to failed ones
                    if ($v->status == 'success' && in_array(strtolower($v->kind), ['sale', 'capture'])) {

                        // 19/05/2020 | CB | Dev 49588 | Get payment description from lookup if set
                        $payment_description = $payment_method_lookup->getLookupValueIfExists($v->gateway);

                        $order_details->payments[] = array('description' => $payment_description, 'amount' => $v->amount);
                    }
                }
            }
        } catch (Exception $e) {
            // we need to ignore this order as with no transaction we have no payment details
            // We could potentially return an error to OrderWise if required
            continue;
        }

        // Basics
        $order_details->order_id = $order_row->id;
        $order_details->order_date = $order_row->created_at;
        $order_details->customer_order_ref = $order_row->order_number;

        // 04/07/2019 | DS | Bug 59530 | Clear the old address data so we don't populate address data for an order that doesn't have an address
        $s_addr = new stdClass();

        // Customer
        // Billing, shipping & customer details are all optional so we find the first one
        // populated in order or preference
        if (isset($order_row->billing_address)) {
            $s_addr = $order_row->billing_address;
        } else {
            if (isset($order_row->customer->default_address)) {
                $s_addr = $order_row->customer->default_address;
            } else {
                if (isset($order_row->shipping_address)) {
                    $s_addr = $order_row->shipping_address;
                }
            }
        }
        $order_details->customer_currency_code = $order_row->currency;
        $order_details->customer_account = $order_row->customer->id;
        $order_details->customer['statement_addr']->salutation = "";
        $order_details->customer['statement_addr']->name = $format->FormatName($s_addr->first_name . ' ' . $s_addr->last_name);
        $order_details->customer['statement_addr']->line1 = $format->FormatAddress($s_addr->address1);
        $order_details->customer['statement_addr']->line2 = $format->FormatAddress($s_addr->address2);
        $order_details->customer['statement_addr']->town = str_replace('— Please Select —', '', $format->FormatAddress($s_addr->city));
        $order_details->customer['statement_addr']->county = str_replace('— Please Select —', '', $format->FormatAddress($s_addr->province));
        $order_details->customer['statement_addr']->postcode = $format->FormatPostcode($s_addr->zip);
        $order_details->customer['statement_addr']->country = $s_addr->country;
        $order_details->customer['statement_addr']->countryiso = $s_addr->country_code;
        $order_details->customer['statement_addr']->telephone = $format->FormatTelephone($s_addr->phone);
        $order_details->customer['statement_addr']->fax = '';
        $order_details->customer['statement_addr']->email = $order_row->email;
        $order_details->customer['invoice_addr'] = clone $order_details->customer['statement_addr'];

        // Delivery
        $order_details->customer['delivery_addr']->name = $format->FormatName($order_row->shipping_address->company);
        $order_details->customer['delivery_addr']->contact = $format->FormatName($order_row->shipping_address->first_name . ' ' . $order_row->shipping_address->last_name);
        $order_details->customer['delivery_addr']->line1 = $format->FormatAddress($order_row->shipping_address->address1);
        $order_details->customer['delivery_addr']->line2 = $format->FormatAddress($order_row->shipping_address->address2);
        $order_details->customer['delivery_addr']->town = str_replace('— Please Select —', '', $format->FormatAddress($order_row->shipping_address->city));
        $order_details->customer['delivery_addr']->county = str_replace('— Please Select —', '', $format->FormatAddress($order_row->shipping_address->province));
        $order_details->customer['delivery_addr']->postcode = $format->FormatPostcode($order_row->shipping_address->zip);
        $order_details->customer['delivery_addr']->country = $order_row->shipping_address->country;
        $order_details->customer['delivery_addr']->countryiso = $order_row->shipping_address->country_code;
        $order_details->customer['delivery_addr']->telephone = $format->FormatTelephone($order_row->shipping_address->phone);
        $order_details->customer['delivery_addr']->email = $order_row->email;

        // Contact
        $order_details->customer['contact']->email = $order_row->customer->email;
        $order_details->customer['contact']->telephone = $order_row->customer->phone;

        // The xml block contains multiple shipping methods so we use the first one
        // 19/05/2020 | CB | Dev 49588 | Use delivery method from lookup if matches
        $order_details->delivery_method = $delivery_method_lookup->getLookupValueIfExists($order_row->shipping_lines[0]->code);
        $order_details->order_special_instructions = $order_row->note;
        $order_details->delivery_special_instructions = '';

        // Shipping
        $shipping_tax = 0;
        $shipping_tax_rate = 0;
        if (count($order_row->shipping_lines) > 0) {
            foreach ($order_row->shipping_lines as $v) {
                $order_details->delivery_gross += $v->price;
                if (count($v->tax_lines) > 0) {
                    foreach ($v->tax_lines as $t) {
                        if ($shopify->isTaxTitle($t->title)) {
                            $shipping_tax += $t->price;
                            $shipping_tax_rate += $t->rate;
                        }
                    }
                }
            }
        }
        $order_details->delivery_vat = $shipping_tax;

        if (!$order_row->taxes_included) {
            $order_details->delivery_net = $order_details->delivery_gross;
            $order_details->delivery_gross += $shipping_tax;
        } else {
            $order_details->delivery_net = $order_details->delivery_gross - $shipping_tax;
        }
        if ($order_details->delivery_gross > 0) {
            // 11/09/2020 | CB | Bug 67459 | Use tax amount instead of rate given
            #$order_details->delivery_vat_code = $format->GetVatCode($shipping_tax_rate);
            $order_details->delivery_vat_code = $format->GetVatCode($order_details->delivery_vat);
        }

        // Financials
        $order_details->order_gross = $order_row->total_price;
        $order_details->order_vat = $order_row->total_tax;
        $order_details->order_net = $order_row->total_price - $order_row->total_tax;

        // 01/09/2020 | CB | Bug 67389 | Removed below and now handle discounts at line basis
        // Discounts do not have tax rates/codes or any indication that they're net/gross pricing
        // Therefore tax codes will be set by the session
//        if (count($order_row->discount_codes) > 0) {
//            $applicable_tax = 0;
//            if (count($order_row->tax_lines) > 0) {
//                foreach ($order_row->tax_lines as $v) {
//                    if ($shopify->isTaxTitle($v->title) && $v->price > 0) {
//                        $applicable_tax = 20;
//                    }
//                }
//            }
//            foreach ($order_row->discount_codes as $d) {
//                $amt = $d->amount / (1 + ($applicable_tax / 100));
//                $order_details->discounts[] = array('description' => $d->code, 'price' => $amt, 'tax_code' => $format->GetVatCode($applicable_tax));
//            }
//        }

        // Lines
        $dissur = ['T0' => 0, 'T1' => 0]; // 01/09/2020 | CB | Bug 67389 | Added
        foreach ($order_row->line_items as $line) {
            // The same field is used for net and gross and the taxes_included
            // flag determines what the value represents. There is always a tax
            // array for a line whether tax is applied or not and from this we
            // need to pull the applicable tax - if !taxes_included we need to
            // add this value to the line price to work up the gross
            $line_tax_rate = 0;
            $line_tax = 0;
            foreach ($line->tax_lines as $k => $v) {
                if ($shopify->isTaxTitle($v->title)) {
                    $line_tax_rate += $v->rate;
                    $line_tax += $v->price;
                }
            }
            // 11/09/2020 | CB | Bug 67459 | Use tax amount instead of rate given
            #$tax_code = $format->GetVatCode($line_tax_rate);
            $tax_code = $format->GetVatCode($line_tax);

            /**
            * 11/09/2020 | CB | Bug 67459
             * Changed how net/gross calcs. work. Vat rate and have discount tax removed already
             * which then means the calculated value is wrong. Instead, just work out what it should
             * be ourselves.
             */
            $gross = $line->price;
            $net = $line->price;
            if ($line_tax > 0 && $line_tax_rate > 0) {
                if ($order_row->taxes_included) {
                    #$net = $line->price - ($line_tax / $line->quantity);
                    $net = $gross / (1 + $line_tax_rate);
                } else {
                    #$gross = $line->price + ($line_tax / $line->quantity);
                    $gross = $net + ($net * $line_tax_rate);
                }
            }

            //22/05/2019 | TL | Dev 43368 | Script Edit to handle multiple stock locations | Add Stock Location to the line

            $stock_location_name = "";

            if (!empty($order_row->location_id) && isset($shopify->stock_locations[$order_row->location_id])) {
                $stock_location_name = $shopify->stock_locations[$order_row->location_id]->name;

                // 19/05/2020 | CB | Dev 49588 | Get stock location name to use from lookup if required
                $stock_location_name = $stock_location_lookup->getLookupValueIfExists($stock_location_name);
            }
//dd($order_row);
            // Add the line
            $order_details->order_lines[] = array(
                'ecommerce_code' => $line->sku,
                'variant_code' => $line->sku,
                'external_item_id' => $line->id,
                'quantity' => $line->quantity,
                ($shopify->import_as_gross ? 'item_gross' : 'item_net') => (float) $line->quantity * (float) $line->price == (float) $line->total_discount ? 0.00 : ($shopify->import_as_gross ? $gross : $net),
                'vat_code' => $tax_code,
                'stock_location_id' => $stock_location_name,
            );

            // 11/09/2020 | CB | Bug 67459 | Added
            if (!empty($line->discount_allocations)) {
                foreach($line->discount_allocations as $line_discount_allocation) {
                    if ($line_discount_allocation->amount > 0) {
                        $line_discount = $line_discount_allocation->amount;
                        if ($tax_code == 'T1' && !$shopify->import_discount_as_gross) {
                            $line_discount = $line_discount / 1.2;
                        }
                        $dissur[$tax_code] += $line_discount;
                    }
                }
            }
        }

        // 11/09/2020 | CB | Bug 67459 | Added
        $discount_description = 'WebDiscount';
        if (!empty($order_row->discount_codes) && count($order_row->discount_codes) == 1) {
            $discount_description = !empty($order_row->discount_codes[0]->code) ? $order_row->discount_codes[0]->code :'WebDiscount';
        }

        // 11/09/2020 | CB | Bug 67459 | Added
        foreach ($dissur as $dissur_tax_code => $dissur_amount) {
            if ($dissur_amount != 0) {
                $order_details->discounts[] = array(
                    'description' => $discount_description,
                    'tax_code' => $dissur_tax_code,
                    'price' => round(abs($dissur_amount),2),
                    'gross_discount' => $shopify->import_discount_as_gross,
                );
            }
        }

        // Add the order to the existing xml dom
        $max_orders_reached = false;
        if ($ecom->AppendOrderToXml($order_details)) {
            if ($ecom->GetOrderCounter() >= $ecom->GetMaximumOrdersPerSession() && $ecom->GetMaximumOrdersPerSession() != 0) {
                $max_orders_reached = true;
                break;
            }
        }
    }

    // If we've reached the max required orders flag more_orders
    // as false to drop out of the do...while
    if ($max_orders_reached || $_GET['order'] != '') {
        $more_orders = false;
    } else {
        // If we've reached the max orders required load the next
        // page. The page counter is incremented in the method.
        try {
            $order_list = $shopify->LoadOrders($order_filter);
        } catch (Exception $e) {
            $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
            $ecom->ReturnResponseXML();
        }
        $more_orders = (count($order_list) > 0) ? true : false;
    }
} while ($more_orders);


$ecom->ExportToXml();
